{ --------------------------------------------------------------
                PROJECT: DELIGHT SOFTWARE COMPONENTS

  $Id: _XMLIniFile_xmlworks.pas,v 1.2 2004/07/10 16:04:29 elias Exp $
                   Code type      : Object-Pascal VCL

                   Description:
                     TCustomIniFile descent to load and save
                     settings from XML

                  (c) 2002 by delight software gmbh

  Requirements:
    XMLWorks - get it at http://www.delphihome.com/XML/XMLWorks.zip

  License:
    TXMLIniFile (this unit only) is free for private use.
    For commercial use, please contact us at www.delight.ch

    Redistribution of this source code must retain this original header.

    Additional features and bugfixes on this sourcecode must be
    reported to delight software gmbh. Please send a email to
    developer@delight.ch with full source an a short description
    what you've changed. So we can get the changes and fixes out to
    other users.

  Additional Copyright Information:
    XMLWork is part of the delphi-jedi project. Read more about
    XMLWork and how to use it at http://www.delphihome.com/XML/
    or http://www.xmlworks.de. Please read their license for
    XMLWorks !
  --------------------------------------------------------------
}

unit _XMLIniFile_xmlworks;

interface
uses
  classes,
  IniFiles,
  xmlworks;

type
  TDataCollectionItem = class(TXMLCollectionItem)
  private
    FValue: String;
    FIdent: String;
  published
    property Ident: String read FIdent write FIdent;
    property Value: String read FValue write FValue;
  end;

  TDataCollection = class(TXMLCollection)
  private
  public
  end;

  TSectionCollectionItem = class(TXMLCollectionItem)
  private
    FSectionName: String;
    FSectionData: TDataCollection;
  published
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    property SectionName: String read FSectionName write FSectionName;
    property SectionData: TDataCollection read FSectionData;
  end;

  TSectionCollection = class(TXMLCollection)
  private
  public
  end;

  TXMLIniFile = class(TCustomIniFile)
  private
    FXMLData: TSectionCollection;
    FAutoSave: Boolean;
    function GetSection(Section: String): TSectionCollectionItem;
    function GetData(Sect: TSectionCollectionItem;
      Ident: String; CreateIfNotExists: Boolean): TDataCollectionItem;
  protected
  public
    constructor Create(const FileName: string);
    destructor Destroy; override;

    function ReadString(const Section, Ident, Default: string): string; override;
    procedure WriteString(const Section, Ident, Value: String); override;
    procedure ReadSection(const Section: string; Strings: TStrings); override;
    procedure ReadSections(Strings: TStrings); override;
    procedure ReadSectionValues(const Section: string; Strings: TStrings); override;
    procedure EraseSection(const Section: string); override;
    procedure DeleteKey(const Section, Ident: String); override;
    procedure UpdateFile; override;

    property AutoSave: Boolean read FAutoSave write FAutoSave;

    // merge inifiles. use to migrate "old" ini files to XML inifiles
    class procedure MergeFiles(Source, Dest: TCustomIniFile);
  end;


implementation
uses
  SysUtils;

{ TSectionCollectionItem }

constructor TSectionCollectionItem.Create(Collection: TCollection);
begin
  inherited;
  FSectionData := TDataCollection.Create(TDataCollectionItem);
end;

destructor TSectionCollectionItem.Destroy;
begin
  FSectionData.Free;
  inherited;
end;

{ TXMLIniFile }

constructor TXMLIniFile.Create(const FileName: string);
begin
  inherited Create(FileName);
  FXMLData := TSectionCollection.Create(TSectionCollectionItem);
  if FileExists(FileName) then
    FXMLData.LoadFromFile(FileName);
  AutoSave := true;
end;

procedure TXMLIniFile.DeleteKey(const Section, Ident: String);
var
  Sect: TSectionCollectionItem;
  Data: TDataCollectionItem;
begin
  Sect := GetSection(Section);
  if Sect <> nil then begin
    Data := GetData(Sect, Ident, false);
    if Data <> nil then
      Data.Free;
  end;
end;

destructor TXMLIniFile.Destroy;
begin
  if AutoSave then
    UpdateFile;
  FXMLData.Free;
  inherited;
end;

procedure TXMLIniFile.EraseSection(const Section: string);
var
  Sect: TSectionCollectionItem;
begin
  Sect := GetSection(Section);
  if Sect <> nil then
    Sect.Free;
end;

procedure TXMLIniFile.ReadSection(const Section: string;
  Strings: TStrings);
var
  Sect: TSectionCollectionItem;
  Lf: Integer;
begin
  Strings.Clear;
  Sect := GetSection(Section);
  if Sect <> nil then begin
    for Lf := 0 to Sect.SectionData.Count-1 do begin
      Strings.Add(TDataCollectionItem(Sect.SectionData.Items[Lf]).Ident);
    end;
  end;
end;

procedure TXMLIniFile.ReadSections(Strings: TStrings);
var
  Lf: Integer;
begin
  Strings.Clear;
  for Lf := 0 to FXMLData.Count-1 do
    Strings.Add(TSectionCollectionItem(FXMLData.Items[Lf]).SectionName);
end;

procedure TXMLIniFile.ReadSectionValues(const Section: string;
  Strings: TStrings);
var
  Sect: TSectionCollectionItem;
  Lf: Integer;
begin
  Strings.Clear;
  Sect := GetSection(Section);
  if Sect <> nil then begin
    for Lf := 0 to Sect.SectionData.Count-1 do begin
      Strings.Add(TDataCollectionItem(Sect.SectionData.Items[Lf]).Value);
    end;
  end;
end;

function TXMLIniFile.ReadString(const Section, Ident,
  Default: string): string;
var
  Sect: TSectionCollectionItem;
  Data: TDataCollectionItem;
begin
  Result := '';
  Sect := GetSection(Section);
  if Sect <> nil then
  begin
    Data := GetData(Sect, Ident, false);
    if Data <> nil then
      Result := Data.Value
    else
      Result := Default;  
  end
end;

procedure TXMLIniFile.UpdateFile;
begin
  FXMLData.SaveToFile(FileName);
end;

function TXMLIniFile.GetSection(Section: String): TSectionCollectionItem;
var
  Lf: Integer;
begin
  Result := nil;
  for Lf := 0 to FXMLData.Count-1 do begin
    if TSectionCollectionItem(FXMLData.Items[Lf]).SectionName = Section then begin
      Result := TSectionCollectionItem(FXMLData.Items[Lf]);
      Break;
    end;
  end;
end;

function TXMLIniFile.GetData(Sect: TSectionCollectionItem; Ident: String; CreateIfNotExists: Boolean): TDataCollectionItem;
var
  Lf: Integer;
begin
  Result := nil;
  for Lf := 0 to Sect.SectionData.Count-1 do begin
    if TDataCollectionItem(Sect.SectionData.Items[Lf]).Ident = Ident then begin
      Result := TDataCollectionItem(Sect.SectionData.Items[Lf]);
      Break;
    end;
  end;
  if (Result = nil) and CreateIfNotExists then
    Result := TDataCollectionItem(Sect.SectionData.Add);
end;

procedure TXMLIniFile.WriteString(const Section, Ident, Value: String);
var
  Sect: TSectionCollectionItem;
  Data: TDataCollectionItem;
begin
  if (Section <> '') and (Ident <> '') then begin
    Sect := GetSection(Section);
    if Sect = nil then begin
      Sect := TSectionCollectionItem(FXMLData.Add);
      Sect.SectionName := Section;
    end;
    Data := GetData(Sect, Ident, true);
    Data.Ident := Ident;
    Data.Value := Value;
  end else
    raise Exception.Create('TXMLIniFile.WriteString: try to write empty strings!');
end;

class procedure TXMLIniFile.MergeFiles(Source, Dest: TCustomIniFile);
var
  Lf, Lfs: Integer;
  Sections, Section: TStrings;
begin
  Assert(Assigned(Source) and Assigned(Dest));

  Sections := TStringList.Create;
  Section := TStringList.Create;
  try
    Source.ReadSections(Sections);
    for Lf := 0 to Sections.Count-1 do begin
      Source.ReadSection(Sections[Lf], Section);
      for Lfs := 0 to Section.Count-1 do begin
        Dest.WriteString(Sections[Lf], Section[Lfs], Source.ReadString(Sections[Lf], Section[Lfs], ''));
      end;
    end;
  finally
    Sections.Free;
    Section.Free;
  end;
end;

end.

