{ --------------------------------------------------------------
                PROJECT: DELIGHT SOFTWARE COMPONENTS

  $Id: _XMLIniFile_openxml.pas,v 1.7 2007/11/21 09:24:32 elias Exp $
                   Code type      : Object-Pascal VCL

                   Description:
                     TCustomIniFile descent to load and save
                     settings from XML

                  (c) 2004 by delight software gmbh

  Requirements:
    XDOM 2.4.0 - get it at http://www.philo.de/

  License:
    _XMLIniFile_openxml (this unit only) is free for private use.
    For commercial use, please contact us at www.delight.ch

    Redistribution of this source code must retain this original header.

    Additional features and bugfixes on this sourcecode must be
    reported to delight software gmbh. Please send a email to
    developer@delight.ch with full source an a short description
    what you've changed. So we can get the changes and fixes out to
    other users.

  Encryption:
    For encryption the DCPcrypt library V1.3 with CFORM defined is uses.
    http://www.scramdisk.clara.net/

  Additional Copyright Information:
    Please have a look at XDOM_2_4.pas header for more informations
    about OpenXML license!
  --------------------------------------------------------------
}

unit _XMLIniFile_openxml;

// use this for backward compatibility with the "old" TXMLIniFile which has based on xmlworks
// if this define is defined the "old" xmlworks files got merged into this new format.
{$DEFINE MERGEOLD}
{$DEFINE USEDCPCRYPT}

interface
uses
  classes,
  IniFiles,
{$IFDEF MERGEOLD} _XMLIniFile_xmlworks, {$ENDIF MERGEOLD}
{$IFDEF USEDCPCRYPT} DCPcrypt, {$ENDIF USEDCPCRYPT}
  XDOM_3_1;

type
  {$IFDEF USEDCPCRYPT}
  TDCP_blockcipherClass = class of TDCP_blockcipher;
  {$ENDIF USEDCPCRYPT}

  TXMLIniFile = class(TCustomIniFile)
  private
    FDomParser: TXmlToDomParser;
    FDomImplementation: TDomImplementation;
    FDocument: TdomDocument;
    FAutoSave: Boolean;
    FModified: Boolean;
    {$IFDEF USEDCPCRYPT}
    FCryptKey: String;
    FDCPCipherClass: TDCP_blockcipherClass;    
    {$ENDIF USEDCPCRYPT}
    procedure SubCreate;
    function FileToDom(const AFileName: String): TdomDocument;
  protected
    function FindElementByName(const AElement: TdomElement; const AName: String): TdomElement;
    function SectionList: TdomElement;
    function GetSection(const ASection: String): TdomElement;
  public
    {$IFDEF USEDCPCRYPT}
    constructor Create(const FileName, ACryptKey: string; ACPCipherClass: TDCP_blockcipherClass); overload;
    {$ENDIF USEDCPCRYPT}
    constructor Create(const FileName: string); {$IFDEF USEDCPCRYPT} overload; {$ENDIF USEDCPCRYPT}
    destructor Destroy; override;

    function ReadString(const Section, Ident, Default: string): string; override;
    procedure WriteString(const Section, Ident, Value: String); override;
    procedure ReadSection(const Section: string; Strings: TStrings); override;
    procedure ReadSections(Strings: TStrings); override;
    procedure ReadSectionValues(const Section: string; Strings: TStrings); override;
    procedure EraseSection(const Section: string); override;
    procedure DeleteKey(const Section, Ident: String); override;
    procedure UpdateFile; override;

    property AutoSave: Boolean read FAutoSave write FAutoSave;
    property Modified: Boolean read FModified write FModified;

    {$IFDEF USEDCPCRYPT}
    property CryptKey: String read FCryptKey write FCryptKey;
    property DCPCipherClass: TDCP_blockcipherClass read FDCPCipherClass write FDCPCipherClass;
    {$ENDIF USEDCPCRYPT}

    // merge inifiles. use to migrate "old" ini files to XML inifiles
    class procedure MergeFiles(Source, Dest: TCustomIniFile);
  end;


implementation
uses
  UriUtils,
  SysUtils;

{ TXMLIniFile }


class procedure TXMLIniFile.MergeFiles(Source, Dest: TCustomIniFile);
var
  Lf, Lfs: Integer;
  Sections, Section: TStrings;
begin
  Assert(Assigned(Source) and Assigned(Dest));

  Sections := TStringList.Create;
  Section := TStringList.Create;
  try
    Source.ReadSections(Sections);
    for Lf := 0 to Sections.Count-1 do begin
      Source.ReadSection(Sections[Lf], Section);
      for Lfs := 0 to Section.Count-1 do begin
        Dest.WriteString(Sections[Lf], Section[Lfs], Source.ReadString(Sections[Lf], Section[Lfs], ''));
      end;
    end;
  finally
    Sections.Free;
    Section.Free;
  end;
end;

{$IFDEF USEDCPCRYPT}
constructor TXMLIniFile.Create(const FileName, ACryptKey: string;
  ACPCipherClass: TDCP_blockcipherClass);
begin
  inherited Create(FileName);

  FCryptKey := ACryptKey;
  FDCPCipherClass := ACPCipherClass;

  SubCreate;
end;
{$ENDIF USEDCPCRYPT}

constructor TXMLIniFile.Create(const FileName: string);
begin
  inherited Create(FileName);

  {$IFDEF USEDCPCRYPT}
  FCryptKey := '';
  FDCPCipherClass := nil;
  {$ENDIF USEDCPCRYPT}

  SubCreate;
end;

procedure TXMLIniFile.SubCreate;
var
  {$IFDEF MERGEOLD} oldINI: _XMLIniFile_xmlworks.TXMLIniFile; {$ENDIF MERGEOLD}
  {$IFDEF USEDCPCRYPT} ms: TMemoryStream; bc: TDCP_blockcipher; {$ENDIF USEDCPCRYPT}
  dmy: Integer;
begin
  FModified := false;

  FDomParser := TXmlToDomParser.Create(nil);
  FDomImplementation := TDomImplementation.Create(nil);
  FDomParser.DOMImpl := FDomImplementation;
  FDomParser.KeepEntityRefs := false;

  if FileExists(FileName) then begin
    {$IFDEF USEDCPCRYPT}
    if Assigned(FDCPCipherClass) then begin
      ms := TMemoryStream.Create;
      try
        ms.LoadFromFile(FileName);

        try
          bc := FDCPCipherClass.Create(nil);
          try
            bc.InitStr(CryptKey);
            bc.DecryptCFB(ms.Memory^, ms.Memory^, ms.Size);
            FDocument := FDomParser.parseStream(ms, '', '', nil) as TdomDocument;
          finally
            bc.Free;
          end;
        except
          FDocument := FileToDom(FileName);
        end;
      finally
        ms.Free;
      end;
    end else
    {$ENDIF USEDCPCRYPT}
      FDocument := FileToDom(FileName)
  end else begin
    FDocument := FDomImplementation.createDocument('inifile', nil);
  end;

  {$IFDEF MERGEOLD}
  if FDocument.documentElement.nodeName = 'Section-list' then begin
    oldINI := _XMLIniFile_xmlworks.TXMLIniFile.Create(FileName);
    try
      FDomImplementation.freeDocument(FDocument);
      FDocument := FDomImplementation.createDocument('inifile', nil);
      TXMLIniFile.MergeFiles(oldINI, self);
      FModified := true;
    finally
      oldINI.Free;
    end;
  end;
  {$ENDIF MERGEOLD}

  AutoSave := true;
end;

function TXMLIniFile.FileToDom(const AFileName: String): TdomDocument;
var
  SourceStream: TFileStream;
begin
  // EZ: have made this methode to have fmShareDenyWrite because TXmlToDomParser.fileToDom has not
  if Length(Filename) = 0 then
    raise Exception.create('Filename not specified.');

  SourceStream := TFileStream.Create(AFileName, fmOpenRead or fmShareDenyWrite);
  try
    Result := FDomParser.parseStream(SourceStream, '', FilenameToUriWideStr(AFileName, []), nil) as TdomDocument;
  finally
    SourceStream.free;
  end;
end;

procedure TXMLIniFile.DeleteKey(const Section, Ident: String);
var
  Sect: TdomElement;
  Data: TdomNode;
begin
  Sect := GetSection(Section);
  if Assigned(Sect) then begin
    Data := FindElementByName(Sect, Ident);
    if Assigned(Data) then begin
      Sect.removeChild(Data);
      Data.Free;
      FModified := true;
    end;
  end;
end;

destructor TXMLIniFile.Destroy;
begin
  if AutoSave then
    UpdateFile;

  FDomImplementation.freeDocument(FDocument);
  FDomImplementation.Free;
  FDomParser.Free;

  inherited;
end;

procedure TXMLIniFile.EraseSection(const Section: string);
var
  Sect: TdomNode;
begin
  Sect := GetSection(Section);
  if Assigned(Sect) then begin
    SectionList.removeChild(Sect);
    Sect.Free;
    FModified := true;
  end;
end;

procedure TXMLIniFile.ReadSection(const Section: string;
  Strings: TStrings);
var
  Sect: TdomElement;
  e: TdomElement;
begin
  Strings.Clear;
  Sect := GetSection(Section);
  if Assigned(Sect) then begin
    e := Sect.findFirstChildElement;
    while Assigned(e) do begin
      Strings.Add(e.getAttributeLiteralValue('name'));
      e := e.findNextSiblingElement;
    end;
  end;
end;

procedure TXMLIniFile.ReadSections(Strings: TStrings);
var
  e: TdomElement;
begin
  Strings.Clear;

  e := SectionList.findFirstChildElement;
  while Assigned(e) do begin
    Strings.Add(e.getAttributeLiteralValue('name'));
    e := e.findNextSiblingElement;
  end;
end;

procedure TXMLIniFile.ReadSectionValues(const Section: string;
  Strings: TStrings);
var
  Sect: TdomElement;
  e: TdomElement;
begin
  Strings.Clear;
  Sect := GetSection(Section);
  if Assigned(Sect) then begin
    e := Sect.findFirstChildElement;
    while Assigned(e) do begin
      Strings.Add(e.textContent);
      e := e.findNextSiblingElement;
    end;
  end;
end;

function TXMLIniFile.ReadString(const Section, Ident,
  Default: string): string;
var
  Sect: TdomElement;
  Data: TdomElement;
begin
  Result := '';
  Sect := GetSection(Section);
  if Assigned(Sect) then
  begin
    Data := FindElementByName(Sect, Ident);
    if Assigned(Data) then
      Result := Data.textContent
    else
      Result := Default;
  end
end;

procedure TXMLIniFile.UpdateFile;
var
  DomToXML: TDomToXmlParser;
  fs: TMemoryStream;
  {$IFDEF USEDCPCRYPT} bc: TDCP_blockcipher; {$ENDIF USEDCPCRYPT}
begin
  DomToXML := TDomToXmlParser.Create(nil);
  try
    DomToXML.DOMImpl := FDomImplementation;
    FDocument.normalize;

    fs := TMemoryStream.Create;
    try
      DomToXML.writeToStream(FDocument, 'UTF-8', fs); //'UTF-16BE'

      {$IFDEF USEDCPCRYPT}
      if Assigned(FDCPCipherClass) then begin
        bc := FDCPCipherClass.Create(nil);
        try
          bc.InitStr(CryptKey);
          bc.EncryptCFB(fs.Memory^, fs.Memory^, fs.Size);
        finally
          bc.Free;
        end;
      end;
      {$ENDIF USEDCPCRYPT}

      fs.SaveToFile(FileName);
      FModified := false;
    finally
      fs.Free;
    end;
  finally
    DomToXML.Free;
  end;
end;

procedure TXMLIniFile.WriteString(const Section, Ident, Value: String);
var
  Sect: TdomElement;
  Data: TdomElement;
begin
  if (Section <> '') and (Ident <> '') then begin
    Sect := GetSection(Section);
    if not Assigned(Sect) then begin
      Sect := TdomElement.create(FDocument, 'section');
      Sect.setAttribute('name', Section);
      SectionList.appendChild(Sect);
    end;

    Data := FindElementByName(Sect, Ident);
    if not Assigned(Data) then begin
      Data := TdomElement.create(FDocument, 'data');
      Sect.appendChild(Data);
    end else
      Data.clear;

    Data.setAttribute('name', Ident);
    Data.appendChild(FDocument.createTextNode(Value));
    FModified := true;
  end else
    raise Exception.Create('TXMLIniFile.WriteString: try to write empty strings!');
end;

function TXMLIniFile.GetSection(const ASection: String): TdomElement;
begin
  Result := FindElementByName(SectionList, ASection);
end;

function TXMLIniFile.SectionList: TdomElement;
begin
  Result := FDocument.documentElement;
end;

function TXMLIniFile.FindElementByName(const AElement: TdomElement; const AName: String): TdomElement;
var
  e: TdomElement;
begin
  Result := nil;

  e := AElement.findFirstChildElement;
  while Assigned(e) and not Assigned(Result) do begin
    if e.getAttributeLiteralValue('name') = AName then
      Result := e
    else
      e := e.findNextSiblingElement;
  end;
end;

end.

